from pyparsing import Literal, oneOf

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.evidences import FailedEvidence, ServiceEvidence
from codeable_detectors.js.jsDetectors import detect_js_import
from codeable_detectors.pyparsing_patterns import ID, round_braces_block


class JSExpressService(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False

        import_matches = detect_js_import(ctx, "express")
        if not import_matches:
            return FailedEvidence("JS express import not found")

        for importMatch in import_matches:
            matches.append(importMatch)
            create_app_matches = ctx.matches_pattern(ID + Literal("=") + Literal(importMatch.kwargs["importVar"]) +
                                                     Literal("(") + Literal(")"))
            if not create_app_matches:
                return FailedEvidence("JS express app creation not found")

            for createAppMatch in create_app_matches:
                matches.append(createAppMatch)
                app_var = createAppMatch.text[:createAppMatch.text.find("=")].strip()
                http_handler_matches = ctx.matches_pattern(Literal(app_var) + Literal(".") +
                                                           oneOf("get put post delete head") +
                                                           round_braces_block +
                                                           Literal(";"))

                if http_handler_matches:
                    matches.extend(http_handler_matches)
                    is_detected = True

        if is_detected:
            return ServiceEvidence(matches).set_properties(detector_technology_types=["javascript", "express"],
                                                           kwargs=kwargs)
        return FailedEvidence("Http handler (get, put, post, delete, or head) call on a JS express app not found")
